#include <cstring>
#include <cassert>
#include "enum_hash.hpp"

EnumHashTable::~EnumHashTable() {
  delete [] h;
}

EnumHashTable::EnumHashTable(uint init_nid, uint init_nb, uint _hash_size) :
  id2head(init_nid, UINT_NULL), free_id(init_nid, 0),
  b(init_nb*6, UINT_NULL), free_b(init_nb, 0), hash_size(_hash_size)
{
  uint i = 0;
  h = new uint[hash_size];
  for (; i < hash_size; ++ i)
    h[i] = UINT_NULL;

  i = 0;
  for (list<uint>::iterator it = free_id.begin(); it != free_id.end(); ++ it) {
    *it = i;
    ++i;
  }

  i = 0;
  for (list<uint>::iterator it = free_b.begin(); it != free_b.end(); ++ it) {
    *it = i;
    i += 6;
  }
}

uint EnumHashTable::new_id() {
  if (free_id.size() == 0) {
    // too small; double the size;
    uint s = id2head.size();
    id2head.insert(id2head.end(), s, UINT_NULL);
    for (uint i = s; i < s+s; ++i)
      free_id.push_back(i);    
  }
  uint p = free_id.front();
  free_id.pop_front();
  assert(id2head[p] == UINT_NULL);
  return p;  
}

uint EnumHashTable::new_b() {
  if (free_b.size() == 0) {
    // too small; double the size;
    uint s = b.size();
    b.insert(b.end(), s, UINT_NULL);
    for (uint i = s; i < s+s; i += 6)
      free_b.push_back(i);    
  }
  uint p = free_b.front();
  free_b.pop_front();
  return p;  
}

bool EnumHashTable::del_id(uint k) {
  assert(id2head[k] == UINT_NULL); // does not contain any content
  
  free_id.push_front(k);
  return true;
}

bool EnumHashTable::del_b(uint k) {  
  free_b.push_front(k);
  return true;
}

uint EnumHashTable::find(uint id, uint c, uint& hash) {
  uint idc[2];
  idc[0] = id;
  idc[1] = c;
  hash = SuperFastHash(reinterpret_cast<char* const>(idc), sizeof(uint)*2) % hash_size;
  //uint head = h[hash];
  uint it = h[hash]; //head;
  while (it != UINT_NULL) {
    if (b[it + 4] == id && b[it + 5] == c) {
      return it;
    }
    it = b[it + 3];
  }
  return UINT_NULL;
}

bool EnumHashTable::insert_(uint id, uint c, uint hash) {
  // need to be called AFTER new_id(id)
  uint b_it = new_b();
  b[b_it + 4] = id;
  b[b_it + 5] = c;

  b[b_it] = UINT_NULL;
  b[b_it + 1] = id2head[id];
  if (id2head[id] != UINT_NULL)
    b[id2head[id]] = b_it;
  id2head[id] = b_it;

  /*
  uint idc[2];
  idc[0] = id;
  idc[1] = c;
  uint hash = SuperFastHash(reinterpret_cast<char* const>(idc), sizeof(uint)*2) % hash_size;
  */

  b[b_it + 2] = UINT_NULL;
  b[b_it + 3] = h[hash];
  if (h[hash] != UINT_NULL)
    b[h[hash] + 2] = b_it;
  h[hash] = b_it;

  return true;
}


bool EnumHashTable::insert_no_duplicate(uint id, uint c) {
  // need to be called AFTER new_id(id)
  uint idc[2];
  idc[0] = id;
  idc[1] = c;
  uint hash = SuperFastHash(reinterpret_cast<char* const>(idc), sizeof(uint)*2) % hash_size;
  //uint head = h[hash];
  uint it = h[hash]; //head;
  while (it != UINT_NULL) {
    if (b[it + 4] == id && b[it + 5] == c) {
      // found copy; return not inserted.
      return false;
    }
    it = b[it + 3];
  }
  
  // no previous copy; insert.
  uint b_it = new_b();
  b[b_it + 4] = id;
  b[b_it + 5] = c;

  b[b_it] = UINT_NULL;
  b[b_it + 1] = id2head[id];
  if (id2head[id] != UINT_NULL)
    b[id2head[id]] = b_it;
  id2head[id] = b_it;
  
  b[b_it + 2] = UINT_NULL;
  b[b_it + 3] = h[hash];
  if (h[hash] != UINT_NULL)
    b[h[hash] + 2] = b_it;
  h[hash] = b_it;

  return true;
}


bool EnumHashTable::delete_(uint b_it) {
  // delete from id2head
  if (b[b_it] == UINT_NULL) {
    id2head[b[b_it + 4]] = b[b_it + 1];
  }else{
    b[b[b_it] + 1] = b[b_it + 1];
  }
  if (b[b_it + 1] != UINT_NULL) {
    b[b[b_it + 1]] = b[b_it];
  }
  // delete from hash_table
  if (b[b_it + 2] == UINT_NULL) {
    uint idc[2];
    idc[0] = b[b_it + 4];
    idc[1] = b[b_it + 5];
    uint hash = SuperFastHash(reinterpret_cast<char* const>(idc), sizeof(uint)*2) % hash_size;
    h[hash] = b[b_it + 3];
  }else{
    b[b[b_it+2] + 3] = b[b_it + 3];
  }
  if (b[b_it + 3] != UINT_NULL) {
    b[b[b_it + 3] + 2] = b[b_it + 2];
  }
  // del b_it
  free_b.push_front(b_it);

  return true;
}

bool EnumHashTable::clear(uint id) {
  uint b_it = id2head[id];
  uint this_b_it = 0;
  while (b_it != UINT_NULL) {
    this_b_it = b_it;
    b_it = b[b_it + 1];
    delete_(this_b_it);
  }
  return true;
}


void EnumHashTable::move_from_id_to_id(uint from_id, uint to_id) {
  uint bit = id2head[from_id];
  uint this_bit;
  while (bit != UINT_NULL) {
    this_bit = bit;
    bit = b[bit + 1];
    insert_no_duplicate(to_id, b[this_bit + 5]);
    delete_(this_bit);
  }
}


void EnumHashTable::print_id_all_c(uint id) {
  uint bit = id2head[id];
  while (bit != UINT_NULL) {
    printf(" %u", b[bit + 5]);
    bit = b[bit +1];
  }
}

/*
inline uint EnumHashTable::head(uint id) {
  return id2head[id];
}

inline uint EnumHashTable::bit2c(uint b_it) {
  return b[b_it + 5];
}

inline uint EnumHashTable::bit2next(uint b_it) {
  return b[b_it + 3];
}

inline uint EnumHashTable::bit2right(uint b_it) {
  return b[b_it + 1];
}
*/
